// background.js —— Manifest V3 | 企微推送专用 | 完整修复版
console.log('[DTSS] Background service worker loaded.');

/**
 * 发送自动提交消息（来自“创建反馈单”抽屉）
 */
async function sendTableMessage(data) {
  const { webhookUrl } = await chrome.storage.sync.get(['webhookUrl']);
  if (!webhookUrl) {
    console.warn('[Background] Webhook URL 未配置');
    return { success: false, error: 'MISSING_WEBHOOK' };
  }

  const lines = [
    '## <font color="info">报告已发送，请留意查看更新状态</font>',
    '',
    `**底盘号：** ${data.vin || '无'}`,
    `**车牌号：** ${data.plate || '无'}`,
    `**备件号（仅参考）：** ${data.partNo || '无'}`,
    `**用户描述：** ${data.faultDesc || '无'}`,
    `**服务顾问：** ${data.advisor || '无'}`,
    ''
  ];

  const content = lines.join('\n');
  const body = JSON.stringify({
    msgtype: 'markdown',
    markdown: { content }
  });

  try {
    const response = await fetch(webhookUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'text/plain' },
      body: body
    });

    // 企微 webhook 成功时返回 {"errcode":0,"errmsg":"ok"}
    const result = await response.json();
    if (result.errcode === 0) {
      console.log('[Background] 表格消息已成功推送至企微');
      return { success: true };
    } else {
      console.error('[Background] 企微 API 业务错误:', result);
      return { success: false, error: 'WECHAT_API_ERROR', details: result.errmsg };
    }
  } catch (err) {
    console.error('[Background] 网络请求失败:', err);
    return { success: false, error: 'NETWORK_ERROR', details: err.message };
  }
}

/**
 * 发送手动选中方案（来自“交流历史”行操作）
 */
async function sendManualTableMessage(data) {
  const { webhookUrl } = await chrome.storage.sync.get(['webhookUrl']);
  if (!webhookUrl) {
    console.warn('[Background] Webhook URL 未配置');
    return { success: false, error: 'MISSING_WEBHOOK' };
  }

  const lines = [
    '## <font color="warning"> 报告已批复，请及时处理</font>',
    '',
    `**底盘号：** ${data.vin || '无'}`,
    `**车牌号：** ${data.plate || '无'}`,
    `**备件号（仅参考）：** ${data.partNo || '无'}`,
    `**用户描述：** ${data.faultDesc || '无'}`,
    `**处理方案：** ${data.selectedMessage || '无'}`,
    `**服务顾问：** ${data.advisor || '无'}`,
    ''
  ];

  const content = lines.join('\n');
  const body = JSON.stringify({
    msgtype: 'markdown',
    markdown: { content }
  });

  try {
    const response = await fetch(webhookUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'text/plain' },
      body: body
    });

    const result = await response.json();
    if (result.errcode === 0) {
      console.log('[Background] 手动选中方案已推送');
      return { success: true };
    } else {
      console.error('[Background] 企微 API 业务错误 (manual):', result);
      return { success: false, error: 'WECHAT_API_ERROR', details: result.errmsg };
    }
  } catch (err) {
    console.error('[Background] 手动推送网络失败:', err);
    return { success: false, error: 'NETWORK_ERROR', details: err.message };
  }
}

// ================== 消息监听器（唯一入口）==================
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  let handlerPromise;

  if (request.action === 'sendTableMessage') {
    handlerPromise = sendTableMessage(request.data);
  } else if (request.action === 'sendManualTableMessage') {
    handlerPromise = sendManualTableMessage(request.data);
  } else {
    // 未知 action
    sendResponse({ success: false, error: 'UNKNOWN_ACTION' });
    return false;
  }

  // 异步处理并返回结果
  handlerPromise
    .then(result => {
      sendResponse(result);
    })
    .catch(err => {
      console.error('[Background] 消息处理器异常:', err);
      sendResponse({ success: false, error: 'INTERNAL_ERROR' });
    });

  return true; // 保持消息通道开放（必需用于异步响应）
});

// ========== 6. 插件版本更新检查（MV3 安全版）==========
let hasShownUpdateNotification = false;

async function checkForPluginUpdate() {
  const manifest = chrome.runtime.getManifest();
  const currentVersion = manifest.version;
  const versionCheckUrl = `https://www.maiyauto.cn/plugin/version_DTSS.json?v=${encodeURIComponent(currentVersion)}`;

  try {
    const response = await fetch(versionCheckUrl, { cache: 'no-cache' });
    if (!response.ok) {
      console.warn('[派工助手] 版本检查失败:', response.status, response.statusText);
      return;
    }

    //const remote = await response.json();
    let remote;
    try {
      remote = await response.json();
    } catch (parseErr) {
      console.warn('[派工助手] 无效的 JSON 响应:', parseErr);
      return;
    }

    // ✅ 保存最新远程版本（供 popup 或其他模块使用）
    if (remote?.current_version) {
      await chrome.storage.local.set({ latestRemoteVersion: remote.current_version });
    }

    // 检查是否需要更新
    if (!remote?.current_version || remote.current_version === currentVersion) {
      return;
    }

    // 读取历史提醒记录
    const result = await chrome.storage.local.get(['updateNotifyRecord']);
    const record = result.updateNotifyRecord || { version: '', count: 0 };

    const targetVersion = remote.current_version;
    let shouldNotify = false;
    let newCount = record.count;

    if (record.version === targetVersion) {
      if (record.count < 3) {
        newCount = record.count + 1;
        shouldNotify = true;
      }
    } else {
      newCount = 1;
      shouldNotify = true;
    }

    if (shouldNotify) {
      // 保存新记录
      await chrome.storage.local.set({
        updateNotifyRecord: { version: targetVersion, count: newCount }
      });

      // 设置 badge 提醒
      chrome.action.setBadgeText({ text: '!' });
      chrome.action.setBadgeBackgroundColor({ color: '#ff6b35' });

      // 创建通知（无按钮，带 url 跳转）
        const notificationId = `plugin_update_v${targetVersion.replace(/\./g, '_')}_${Date.now()}`;
        const jumpUrl = remote.download_url || 'https://www.maiyauto.cn/plugin/help.html';

        chrome.notifications.create(notificationId, {
          type: 'basic',
          title: '【DTSS推送助手】发现新版本！点击窗口下载！',
          message: `当前版本：${currentVersion} → 新版本：${targetVersion}\n这是第 ${newCount} 次提醒。${remote.release_notes || ''}`,
          iconUrl: 'icon-48.png',
          url: jumpUrl  // ✅ 点击通知自动打开此链接
          requireInteraction: true  // 关键！通知不会自动消失，需要用户点击有交互才行
        });
    }
  } catch (err) {
    console.warn('[派工助手] 检查更新时出错:', err);
  }
}
<!--
// ========== 通知交互处理 ==========
function clearUpdateBadge() {
  chrome.action.setBadgeText({ text: '' });
}

// 点击通知主区域
chrome.notifications.onClicked.addListener((notificationId) => {
  if (notificationId.startsWith('plugin_update')) {
    clearUpdateBadge();
    chrome.tabs.create({ url: 'https://www.maiyauto.cn/plugin/help.html' });
  }
});

// 点击“立即下载”按钮
chrome.notifications.onButtonClicked.addListener((notificationId, buttonIndex) => {
  if (notificationId.startsWith('plugin_update') && buttonIndex === 0) {
    clearUpdateBadge();
    chrome.tabs.create({ url: 'https://www.maiyauto.cn/plugin/help.html' });
  }
});
-->
// ========== 启动 & 定时检查 ==========
// 首次启动检查
checkForPluginUpdate();

// 每 2 小时检查一次（使用 alarms，后台更省电）
chrome.alarms.create('checkPluginUpdate', {
  periodInMinutes: 120
});

chrome.alarms.onAlarm.addListener((alarm) => {
  if (alarm.name === 'checkPluginUpdate') {
    checkForPluginUpdate();
  }
});
